'use client';

import React, { useState } from 'react';
import Select, { SingleValue, StylesConfig } from 'react-select';


interface OptionType {
  value: string;
  label: string;
}

const categoryOptions: OptionType[] = [
  { value: '', label: 'Category' },
  { value: 'tech', label: 'Technology' },
  { value: 'health', label: 'Healthcare' },
  { value: 'finance', label: 'Finance' },
];

const countryOptions: OptionType[] = [
  { value: '', label: 'Country' },
  { value: 'us', label: 'United States' },
  { value: 'uk', label: 'United Kingdom' },
  { value: 'in', label: 'India' },
];

const customStyles: StylesConfig<OptionType, false> = {
  option: (base, state) => ({
    ...base,
    backgroundColor: state.isSelected
      ? '#f3f3f3'
      : state.isFocused
      ? '#f6f6f6'
      : 'white',
    color: '#323232',
    padding: 10,
    cursor: 'pointer',
    fontSize: '14px',
    fontWeight: state.isSelected ? '600' : '400',
  }),
  control: (base, state) => ({
    ...base,
    width: 230,
    color: '#8794a4',
    fontWeight: 700,
    fontSize: 14,
    padding: '2px 5px',
    border: '1px solid #d6d6d6',
    borderRadius: 5,
    backgroundColor: 'transparent',
    boxShadow: state.isFocused ? '0 0 0 1px #d6d6d6' : 'none',
    '&:hover': {
      borderColor: '#d6d6d6',
    },
  }),
  singleValue: (base) => ({
    ...base,
    maxWidth: '100%',
    overflow: 'hidden',
    textOverflow: 'ellipsis',
    whiteSpace: 'nowrap',
    color: '#8794a4',
    marginLeft: '2px',
    marginRight: '2px',
    fontWeight: 700,
  }),
  dropdownIndicator: (base) => ({
    ...base,
    color: '#999',
  }),
  indicatorSeparator: () => ({
    display: 'none',
  }),
};

export default function HeaderSearch() {
  const [selectedCategory, setSelectedCategory] = useState<OptionType>(categoryOptions[0]);
  const [selectedCountry, setSelectedCountry] = useState<OptionType>(countryOptions[0]);

  const handleCategoryChange = (option: SingleValue<OptionType>) => {
    if (option) setSelectedCategory(option);
  };

  const handleCountryChange = (option: SingleValue<OptionType>) => {
    if (option) setSelectedCountry(option);
  };

  return (
    <>
    <section className="header-search max-xl:pt-[140px] max-xl:pb-[50px] py-[50px]">
      <div className="container mx-auto px-4">
        <div className="section-wraper flex flex-wrap max-xl:items-center max-xl:justify-around justify-between gap-y-5 gap-x-[10px]">
          <h1 className="font-extrabold max-xl:text-[40px] max-xxl:text-[45px] 2xl:text-[54px] text-center">
            News & Updates
          </h1>

          <div className="filter flex flex-wrap items-center gap-[10px]">
            <div className="list-category">
              <Select
                options={categoryOptions}
                value={selectedCategory}
                onChange={handleCategoryChange}
                classNamePrefix="react-select"
                placeholder="Category"
                styles={customStyles}
              />
            </div>

            <div className="list-country">
              <Select
                options={countryOptions}
                value={selectedCountry}
                onChange={handleCountryChange}
                classNamePrefix="react-select"
                placeholder="Country"
                styles={customStyles}
              />
            </div>

            <div className="news-search relative">
              <input
                className="search-input w-full bg-transparent font-bold max-w-[230px] rounded-[5px] border border-[#d6d6d6] 
                  h-[42px] text-[#8794a4] pl-[18px] pr-[40px] focus:border-[#d6d6d6] focus-visible:border-[#d6d6d6] focus:outline-none"
                type="text"
                placeholder="Search"
              />
              <span className="search-icon absolute right-3 top-1/2 -translate-y-1/2 text-[#8794a4]">
                <i className="icon-search" />
              </span>
            </div>
          </div>
        </div>
      </div>
    </section>
    </>
  );
}
