'use client';

import React, { useState } from "react";
import Select, { StylesConfig } from 'react-select';

interface OptionType {
      value: string;
      label: string;
  }

const sortOptions = [
  { value: "relevant", label: "Most relevant" },
  { value: "recent", label: "Most recent" },
  { value: "highest", label: "Highest rating" },
  { value: "lowest", label: "Lowest rating" },
];

const filterOptions = [
  "With Photos",
  "5-Star Only",
  "Verified Users",
];

const customStyles: StylesConfig<OptionType, false> = {
      option: (base, state) => ({
        ...base,
        backgroundColor: state.isSelected
        ? '#f3f3f3'
        : state.isFocused
        ? '#f6f6f6'
        : 'white',
        color: '#343434',
        padding: 10,
        cursor: 'pointer',
        fontSize: '14px',
        fontWeight: state.isSelected ? '700' : '400',
    }),
      control: (base, state) => ({
        ...base,
        width: '100%',
        color: '#343434',
        fontWeight: 700,
        fontSize: 16,
        padding: 0,
        border: '0px solid #e8e8e8',
        borderRadius: 0,
        backgroundColor: 'transparent',
        boxShadow: state.isFocused ? '0 0 0 0px #e8e8e8' : 'none',
        '&:hover': {
          borderColor: 'transparent',
      },
      // Responsive font size
  '@media (max-width: 991px)': {
    fontSize: 14,
  },
  }),
      singleValue: (base) => ({
        ...base,
        maxWidth: '100%',
        overflow: 'hidden',
        textOverflow: 'ellipsis',
        whiteSpace: 'nowrap',
        color: '#343434',
        marginLeft: 0,
        padding: 0,
        fontWeight: 700,
    }),
      placeholder: (base) => ({
        ...base,
        display: 'none', // Hides the placeholder (heading-like text)
    }),
      dropdownIndicator: (base, state) => ({
        ...base,
        padding: 0,
        color: '#999',
        transform: state.selectProps.menuIsOpen ? 'rotate(180deg)' : 'rotate(0deg)',
        transition: 'transform 0.2s ease',
        svg: {
          fill: '#999',
          stroke: '#999',
          strokeWidth: 0,
      },
  }),
      indicatorSeparator: () => ({
        display: 'none',
    }),
  };

export default function ReviewsSortHolder() {
  const [selectedSort, setSelectedSort] = useState(sortOptions[0]);
  const [showFilters, setShowFilters] = useState(false);
  const [activeFilters, setActiveFilters] = useState<string[]>([]);

  const toggleFilter = (filter: string) => {
    setActiveFilters((prev) =>
      prev.includes(filter)
        ? prev.filter((f) => f !== filter)
        : [...prev, filter]
    );
  };

  return (
    <div className="reviews-sort-holder flex justify-between items-center pt-[35px] pb-[15px]">
      {/* Sort Dropdown */}
      <div className="sort flex items-center gap-0">
        <span className="text-[#6a6a6a] text-sm font-normal leading-[1.07]">Sort by:</span>
        <div className="w-[150px]">
          <Select
            options={sortOptions}
            value={selectedSort}
            onChange={(opt) => opt && setSelectedSort(opt)}
            classNamePrefix="rs"
            isSearchable={false}
            styles={customStyles}
          />
        </div>
      </div>

      {/* Filter Dropdown */}
      <div className="filter">
      <div className="dropdown relative">
        <button
          onClick={() => setShowFilters((prev) => !prev)}
          className="filter-action flex items-center gap-[5px] text-[#343434] text-base font-bold leading-[1.47] "
        >
          <i className="icon-filter text-[24px] text-[#aea6c5]" /> Filter
          
        </button>

        {showFilters && (
          <ul className="dropdown-menu dropdown-menu-end absolute right-0 mt-1 w-48 !bg-white border border-[#ececec] rounded-[5px] shadow-md z-10">
            {filterOptions.map((filter) => (
              <li key={filter}>
                <button
                  onClick={() => toggleFilter(filter)}
                  className={`w-full text-left px-[15px] py-[7px] text-sm hover:bg-[#f8f9fa] ${
                    activeFilters.includes(filter) ? "bg-gray-100 font-semibold" : ""
                  }`}
                >
                  {filter}
                </button>
              </li>
            ))}
          </ul>
        )}
      </div>
    </div>
    </div>
  );
}
